//
// Base on Microsoft "cpuid.cpp"
//
#include "stdafx.h"
#include <windows.h>
#include <mmsystem.h>
#include "RT_CPUID.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// These are the bit flags that get set on calling cpuid with register eax set to 1
// Feature, edx
#define _MMX_FEATURE_BIT        0x00800000
#define _SSE_FEATURE_BIT        0x02000000
#define _SSE2_FEATURE_BIT       0x04000000
#define _HT_FEATURE_BIT			0x10000000
#define _TM_FEATURE_BIT			0x20000000
#define _IA64_FEATURE_BIT		0x40000000
// Feature2, ecx
#define _SSE3_FEATURE_BIT		0x00000001

// This bit is set when cpuid is called with
// register set to 80000001h (only applicable to AMD)
#define _3DNOW_FEATURE_BIT		0x80000000
//
#define _3DNOW_P_FEATURE_BIT	0x40000000
#define _MMX_P_FEATURE_BIT		0x00400000

CProcessorInfo::CProcessorInfo()
{
    CPU_Info.VendorName.Empty();
    CPU_Info.ModelName.Empty();
    CPU_Info.Family = 0;
    CPU_Info.Model = 0;
    CPU_Info.BrandID = 0;
    CPU_Info.Stepping = 0;
    CPU_Info.Feature = 0;
    CPU_Info.OS_Support = 0;
    CPU_Info.Checks = 0;
	CPU_Info.L1CacheCode = 0;
	CPU_Info.L1CacheData = 0;
	CPU_Info.L2Cache = 0;
	CPU_Info.L3Cache = 0;
	CPU_Info.Speed = 0;
}

CProcessorInfo::~CProcessorInfo()
{
}

void CProcessorInfo::Initial(bool IsDetectSpeed)
{
    DWORD dwStandard = 0;
    DWORD dwStandard2 = 0;
    DWORD dwFeature = 0;
    DWORD dwFeature2 = 0;
    DWORD dwMax = 0;
    DWORD dwExt = 0;
	DWORD dwExtendedLevel = 0;
    int Feature = 0;
    int OS_Support = 0;
    union
	{
        BYTE VendorBuffer[12];
        struct
		{
            DWORD dw0;
            DWORD dw1;
            DWORD dw2;
        } s;
    } Ident;
	//
    if (IsCPUID() == false)   return;
	// Get Standard and Feature
    _asm
	{
        push ebx
        push ecx
        push edx

        // get the vendor string
        xor eax, eax
        cpuid
        mov dwMax, eax
        mov Ident.s.dw0, ebx
        mov Ident.s.dw1, edx
        mov Ident.s.dw2, ecx

        // get the Standard bits
        mov eax, 1
        cpuid
        mov dwStandard, eax
		mov dwStandard2, ebx
		mov dwFeature2, ecx
        mov dwFeature, edx

        // get AMD-specials
        mov eax, 80000000h
        cpuid
        cmp eax, 80000000h
        jc NotAMD
		sub eax, 80000000h
		mov dwExtendedLevel, eax
        mov eax, 80000001h
        cpuid
        mov dwExt, edx
NotAMD:
        pop ecx
        pop ebx
        pop edx
    }
	// MMX
    if (dwFeature & _MMX_FEATURE_BIT)
	{
        Feature |= _CPU_FEATURE_MMX;
        if (IsOS_Support(_CPU_FEATURE_MMX) == true)   OS_Support |= _CPU_FEATURE_MMX;
    }
	// 3DNOW!
    if (dwExt & _3DNOW_FEATURE_BIT)
	{
        Feature |= _CPU_FEATURE_3DNOW;
        if (IsOS_Support(_CPU_FEATURE_3DNOW) == true)   OS_Support |= _CPU_FEATURE_3DNOW;
    }
	// MMX Plus
    if (dwExt & _MMX_P_FEATURE_BIT)
	{
        Feature |= _CPU_FEATURE_MMX_P;
        if (IsOS_Support(_CPU_FEATURE_MMX_P) == true)   OS_Support |= _CPU_FEATURE_MMX_P;
    }
	// 3DNOW! Plus
    if (dwExt & _3DNOW_P_FEATURE_BIT)
	{
        Feature |= _CPU_FEATURE_3DNOW_P;
        if (IsOS_Support(_CPU_FEATURE_3DNOW_P) == true)   OS_Support |= _CPU_FEATURE_3DNOW_P;
    }
	// SSE
    if (dwFeature & _SSE_FEATURE_BIT)
	{
        Feature |= _CPU_FEATURE_SSE;
        if (IsOS_Support(_CPU_FEATURE_SSE) == true)   OS_Support |= _CPU_FEATURE_SSE;
    }
	// SSE2
    if (dwFeature & _SSE2_FEATURE_BIT)
	{
        Feature |= _CPU_FEATURE_SSE2;
        if (IsOS_Support(_CPU_FEATURE_SSE2) == true)   OS_Support |= _CPU_FEATURE_SSE2;
    }
	// SSE3
    if (dwFeature2 & _SSE3_FEATURE_BIT)
	{
        Feature |= _CPU_FEATURE_SSE3;
        if (IsOS_Support(_CPU_FEATURE_SSE3) == true)   OS_Support |= _CPU_FEATURE_SSE3;
    }
	// IA64
    if (dwFeature & _IA64_FEATURE_BIT)
	{
        Feature |= _CPU_FEATURE_IA64;
        if (IsOS_Support(_CPU_FEATURE_IA64) == true)   OS_Support |= _CPU_FEATURE_IA64;
    }
	// HT
    if (dwFeature & _HT_FEATURE_BIT)
	{
        Feature |= _CPU_FEATURE_HT;
        if (IsOS_Support(_CPU_FEATURE_HT) == true)   OS_Support |= _CPU_FEATURE_HT;
    }
	// TM
    if (dwFeature & _TM_FEATURE_BIT)
	{
        Feature |= _CPU_FEATURE_TM;
        if (IsOS_Support(_CPU_FEATURE_TM) == true)   OS_Support |= _CPU_FEATURE_TM;
    }
	//
	CPU_Info.Feature = Feature;
	//
	CPU_Info.OS_Support = OS_Support;
	//
	CPU_Info.Family = ( (dwStandard >> 8) & 0x0F );
	if (CPU_Info.Family == 0x0F)   CPU_Info.Family |= ( (dwStandard >> 16) & 0x0FF0 );
	//
	CPU_Info.Model = ( (dwStandard >> 4) & 0x0F );
	// if (CPU_Info.Model == 15)   CPU_Info.Model |= ( (dwStandard >> 12) & 0x0F );
	//
	CPU_Info.BrandID = ( (dwStandard2) & 0xFF );
	//
	CPU_Info.Stepping = ( (dwStandard) & 0x0F );
	//
	CPU_Info.VendorName.Empty();
	for (int i = 0; i < 12; i++)
	{
		CPU_Info.VendorName.AppendChar( TCHAR(Ident.VendorBuffer[i]) );
	}
	//
	SetModelName();
	//
	CPU_Info.Checks = (_CPU_FEATURE_MMX | _CPU_FEATURE_SSE | _CPU_FEATURE_SSE2 | _CPU_FEATURE_3DNOW);
	//
	if (IsDetectSpeed == true)   SetSpeed();
	//
	SetCache(dwExtendedLevel);
}

bool CProcessorInfo::IsCPUID()
{
    __try
	{
        _asm
		{
            xor eax, eax
            cpuid
        }
    }
    __except (EXCEPTION_EXECUTE_HANDLER)
	{
        return false;
    }
    return true;
}

bool CProcessorInfo::IsOS_Support(unsigned int Feature)
{
    __try
	{
        switch (Feature)
		{
	        case _CPU_FEATURE_SSE:
		        __asm
				{
	                xorps xmm0, xmm0        // executing SSE instruction
		        }
			    break;
	        case _CPU_FEATURE_SSE2:
		        __asm
				{
			        xorpd xmm0, xmm0        // executing SSE2 instruction
				}
	            break;
		    case _CPU_FEATURE_3DNOW:
			    __asm
				{
				    pfrcp mm0, mm0          // executing 3DNow! instruction
					emms
	            }
		        break;
			case _CPU_FEATURE_MMX:
				__asm
				{
					pxor mm0, mm0           // executing MMX instruction
	                emms
		        }
			    break;
			}
		}
    __except (EXCEPTION_EXECUTE_HANDLER)
	{
        if (_exception_code() == STATUS_ILLEGAL_INSTRUCTION)
		{
            return false;
        }
        return false;
    }
    return true;
}

void CProcessorInfo::SetModelName()
{
	CPU_Info.ModelName = _T("Unknown");
	if ( CPU_Info.VendorName == _T("AuthenticAMD") )
	{
		switch (CPU_Info.Family)
		{
			// Am486/AM5x86
	        case 4:
		        CPU_Info.ModelName = _T("AMD Am486");
			    break;
			// K6
	        case 5: 
				switch (CPU_Info.Model)
				{
				    case 0:
					case 1:
					case 2:
					case 3:
		                CPU_Info.ModelName = _T("AMD K5");
				        break;
		            case 6:
				    case 7:
						CPU_Info.ModelName = _T("AMD K6");
		                break;
				    case 8:
						CPU_Info.ModelName = _T("AMD K6-2");
		                break;
				    case 9:
		            case 10:
				    case 11:
		            case 12:
				    case 13:
		            case 14:
				    case 15:
						CPU_Info.ModelName = _T("AMD K6-3");
		                break;
			    }
				break;
			// Athlon
		    case 6:
				switch (CPU_Info.Model)
				{
					case 3:
		                CPU_Info.ModelName = _T("Duron [Spitfire]");
				        break;
					case 4:
						CPU_Info.ModelName = _T("Athlon [Thumderbird]");
				        break;
		            case 6:
						CPU_Info.ModelName = _T("AthlonXP [Paiomino]");
				        break;
				    case 7:
						CPU_Info.ModelName = _T("Duron [Morgan]");
		                break;
				    case 8:
						CPU_Info.ModelName = _T("AthlonXP [Thoroughbred]");
		                break;
		            case 0x0A:
						CPU_Info.ModelName = _T("AthlonXP [Barton]");
		                break;
					default:
						// No model numbers are currently defined
			            CPU_Info.ModelName = _T("Athlon Family");
						break;
			    }
				break;
			// Athlon 64
		    case 0x0F:
	            CPU_Info.ModelName = _T("Athlon 64");
				break;
        }
    }
    else if (CPU_Info.VendorName == _T("GenuineIntel") )
	{
        switch (CPU_Info.Family)
		{
			// 486
		    case 4:
			    switch (CPU_Info.Model)
				{
		            case 0:
			        case 1:
				        CPU_Info.ModelName = _T("INTEL 486DX");
					    break;
		            case 2:
			            CPU_Info.ModelName = _T("INTEL 486SX");
				        break;
					case 3:
						CPU_Info.ModelName = _T("INTEL 486DX2");
						break;
			        case 4:
				        CPU_Info.ModelName = _T("INTEL 486SL");
					    break;
				    case 5:
					    CPU_Info.ModelName = _T("INTEL 486SX2");
						break;
					case 7:
						CPU_Info.ModelName = _T("INTEL 486DX2E");
			            break;
				    case 8:
					    CPU_Info.ModelName = _T("INTEL 486DX4");
						break;
	            }
		        break;
			// 586
			case 5:
				switch (CPU_Info.Model)
				{
		            case 1:
			        case 2:
				    case 3:
					    CPU_Info.ModelName = _T("Pentium");
						break;
		            case 4:
			            CPU_Info.ModelName = _T("Pentium-MMX");
				        break;
				}
	            break;
			// 686
			case 6:
				switch (CPU_Info.Model)
				{
		            case 1:
			            CPU_Info.ModelName = _T("Pentium-Pro");
				        break;
					case 3:
		            case 5:
			            CPU_Info.ModelName = _T("Pentium-II");
				        break;
					case 6:
						CPU_Info.ModelName = _T("Celeron");
						break;
			        case 7:
				    case 8:
						if ((CPU_Info.BrandID % 2) == 0)
							CPU_Info.ModelName = _T("Pentium-III [Coppermine]");
						else
							CPU_Info.ModelName = _T("Celeron [Coppermine 128]");
		                break;
					case 10:
					case 11:
						if ((CPU_Info.BrandID % 2) == 0)
							CPU_Info.ModelName = _T("Pentium-III [Tualatin]");
						else
							CPU_Info.ModelName = _T("Celeron [Tualatin 128]");
		                break;
		        }
			    break;
			// INTEL Pentium-4
		    case 0x0F:
				switch (CPU_Info.Model)
				{
					// 0.18um
					case 0:
						CPU_Info.ModelName = _T("Pentium-4 [Willamette]");
				        break;
		            case 1:
						if (CPU_Info.BrandID == 0x0A)
							CPU_Info.ModelName = _T("Celeron [Willamette 128]");
						else
							CPU_Info.ModelName = _T("Pentium-4 [Willamette]");
				        break;
					// 0.13um
		            case 2:
						switch (CPU_Info.BrandID)
						{
							case 0x08:
								CPU_Info.ModelName = _T("Celeron-4 Mobile");
								break;
							case 0x0E:
								CPU_Info.ModelName = _T("Pentium-4 Mobile");
								break;
							case 0x09:
								CPU_Info.ModelName = _T("Pentium-4 [Northwood]");
								break;
							case 0x0A:
								CPU_Info.ModelName = _T("Celeron-4 [Northwood 128]");
								break;
							case 0x12:
								CPU_Info.ModelName = _T("Celeron-M");
								break;
							case 0x16:
								CPU_Info.ModelName = _T("Pentium-M");
								break;
							default:
								CPU_Info.ModelName = _T("Pentium-4 Family");
								break;
						}
				        break;
					// 0.09um
		            case 3:
						switch (CPU_Info.BrandID)
						{
							case 0x12:
								CPU_Info.ModelName = _T("Celeron-M");
								break;
							case 0x16:
								CPU_Info.ModelName = _T("Pentium-M");
								break;
							default:
								CPU_Info.ModelName = _T("Pentium-4 Family");
								break;
						}
		        }
			    break;
		}
    }
    else if ( CPU_Info.VendorName == _T("CyrixInstead") )
	{
        CPU_Info.ModelName = _T("Cyrix");
    }
    else if ( CPU_Info.VendorName == _T("CentaurHauls") )
	{
        CPU_Info.ModelName = _T("Centaur");
    }
}

bool CProcessorInfo::IsCompatibleFeature(unsigned int Feature)
{
	switch (Feature)
	{
		case _CPU_FEATURE_MMX:
		case _CPU_FEATURE_SSE:
		case _CPU_FEATURE_SSE2:
		case _CPU_FEATURE_3DNOW:
			return ( (CPU_Info.Feature & Feature) && (CPU_Info.OS_Support & Feature) );
		default:
			return false;
	}
}

CString CProcessorInfo::GetFeatureString()
{
	CString Buffer;
	if (CPU_Info.Feature & _CPU_FEATURE_MMX)
	{
		if (CPU_Info.Feature & _CPU_FEATURE_MMX_P)
			Buffer.Append( _T("MMX+, ") );
		else
			Buffer.Append( _T("MMX, ") );
	}
	if (CPU_Info.Feature & _CPU_FEATURE_3DNOW)
	{
		if (CPU_Info.Feature & _CPU_FEATURE_3DNOW_P)
            Buffer.Append( _T("3DNOW!+, ") );
		else
			Buffer.Append( _T("3DNOW!, ") );
	}
	if (CPU_Info.Feature & _CPU_FEATURE_SSE)   Buffer.Append( _T("SSE, ") );
	if (CPU_Info.Feature & _CPU_FEATURE_SSE2)   Buffer.Append( _T("SSE2, ") );
	if (CPU_Info.Feature & _CPU_FEATURE_SSE3)   Buffer.Append( _T("SSE3, ") );
	if (CPU_Info.Feature & _CPU_FEATURE_IA64)   Buffer.Append( _T("IA64, ") );
	if (CPU_Info.Feature & _CPU_FEATURE_HT)   Buffer.Append( _T("HT, ") );
	if (CPU_Info.Feature & _CPU_FEATURE_TM)   Buffer.Append( _T("TM, ") );
	Buffer.AppendFormat( _T("(%u)"), CPU_Info.Feature );
	return Buffer;
}

CString CProcessorInfo::GetOS_SupportString()
{
	CString Buffer;
	if (CPU_Info.OS_Support & _CPU_FEATURE_MMX)
	{
		if (CPU_Info.OS_Support & _CPU_FEATURE_MMX_P)
			Buffer.Append( _T("MMX+, ") );
		else
			Buffer.Append( _T("MMX, ") );
	}
	if (CPU_Info.OS_Support & _CPU_FEATURE_3DNOW)
	{
		if (CPU_Info.OS_Support & _CPU_FEATURE_3DNOW_P)
            Buffer.Append( _T("3DNOW!+, ") );
		else
			Buffer.Append( _T("3DNOW!, ") );
	}
	if (CPU_Info.OS_Support & _CPU_FEATURE_SSE)   Buffer.Append( _T("SSE, ") );
	if (CPU_Info.OS_Support & _CPU_FEATURE_SSE2)   Buffer.Append( _T("SSE2, ") );
	if (CPU_Info.OS_Support & _CPU_FEATURE_SSE3)   Buffer.Append( _T("SSE3, ") );
	if (CPU_Info.OS_Support & _CPU_FEATURE_IA64)   Buffer.Append( _T("IA64, ") );
	if (CPU_Info.OS_Support & _CPU_FEATURE_HT)   Buffer.Append( _T("HT, ") );
	if (CPU_Info.OS_Support & _CPU_FEATURE_TM)   Buffer.Append( _T("TM, ") );
	Buffer.AppendFormat( _T("(%u)"), CPU_Info.OS_Support );
	return Buffer;
}

void CProcessorInfo::SetCache(DWORD ExtendedLevel)
{
	CPU_Info.L1CacheCode = CPU_Info.L1CacheData = CPU_Info.L2Cache = CPU_Info.L3Cache = 0;
	// get processor configuration descriptors
    unsigned __int32 Buffer[4], BufferTemp;
	__asm
	{
		mov		eax, 0x00000002
		cpuid
		mov		Buffer[0], eax
		mov		Buffer[1], ebx
		mov		Buffer[2], ecx
		mov		Buffer[3], edx
	}
	// Reset AL
	Buffer[0] &= 0xFFFFFF00;
	for (int i = 0; i < 3; i++)
	{
		BufferTemp = Buffer[i];
		while (BufferTemp != 0)
		{
			CheckConfig( (BufferTemp & 0xFF) );
			BufferTemp >>= 8;
		}
	}
	// Get Cache by Extended Level
	if (ExtendedLevel >= 6)
	{
	    unsigned __int32	L1CacheCode = 0;
		unsigned __int32	L1CacheData = 0;
		unsigned __int32	L2Cache = 0;
		// L1 CACHE
		__asm
		{
			mov		eax, 0x80000005
			mov		ebx, 0
			mov		ecx, 0
			mov		edx, 0
			cpuid
			shr		ecx, 24
			mov		L1CacheData, ecx
			shr		edx, 24
			mov		L1CacheCode, edx
		}
		CPU_Info.L1CacheData = L1CacheData;
		CPU_Info.L1CacheCode = L1CacheCode;
		// L2 CACHE
		__asm
		{
			mov		eax, 0x80000006
			mov		ebx, 0
			mov		ecx, 0
			mov		edx, 0
			cpuid
			shr		ecx, 16
			mov		L2Cache, ecx
		}
		CPU_Info.L2Cache = L2Cache;
	}
}

void CProcessorInfo::CheckConfig(unsigned int Value)
{
	if (Value == 0)   return;
	switch (Value)
	{
		// L1 Cache Code
		case 0x06:
			CPU_Info.L1CacheCode = 8;
			break;
		case 0x08:
			CPU_Info.L1CacheCode = 16;
			break;
		case 0x15:
			CPU_Info.L1CacheCode = 16;
			break;
		case 0x30:
			CPU_Info.L1CacheCode = 32;
			break;
		case 0x77:
			CPU_Info.L1CacheCode = 16;
			break;
		// L1 Cache Data
		case 0x0A:
			CPU_Info.L1CacheData = 8;
			break;
		case 0x0C:
			CPU_Info.L1CacheData = 16;
			break;
		case 0x10:
			CPU_Info.L1CacheData = 16;
			break;
		case 0x2C:
			CPU_Info.L1CacheData = 32;
			break;
		case 0x60:
			CPU_Info.L1CacheData = 16;
			break;
		case 0x66:
			CPU_Info.L1CacheData = 8;
			break;
		case 0x67:
			CPU_Info.L1CacheData = 16;
			break;
		case 0x68:
			CPU_Info.L1CacheData = 32;
			break;
		// L2 Cache
		case 0x1A:
			CPU_Info.L2Cache = 96;
			break;
		case 0x39:
			CPU_Info.L2Cache = 128;
			break;
		case 0x3B:
			CPU_Info.L2Cache = 128;
			break;
		case 0x3C:
			CPU_Info.L2Cache = 256;
			break;
		case 0x41:
			CPU_Info.L2Cache = 128;
			break;
		case 0x42:
			CPU_Info.L2Cache = 256;
			break;
		case 0x43:
			CPU_Info.L2Cache = 512;
			break;
		case 0x44:
			CPU_Info.L2Cache = 1024;
			break;
		case 0x45:
			CPU_Info.L2Cache = 2048;
			break;
		case 0x78:
			CPU_Info.L2Cache = 1024;
			break;
		case 0x79:
			CPU_Info.L2Cache = 128;
			break;
		case 0x7A:
			CPU_Info.L2Cache = 256;
			break;
		case 0x7B:
			CPU_Info.L2Cache = 512;
			break;
		case 0x7C:
			CPU_Info.L2Cache = 1024;
			break;
		case 0x7D:
			CPU_Info.L2Cache = 2048;
			break;
		case 0x7E:
			CPU_Info.L2Cache = 256;
			break;
		case 0x7F:
			CPU_Info.L2Cache = 512;
			break;
		case 0x81:
			CPU_Info.L2Cache = 128;
			break;
		case 0x82:
			CPU_Info.L2Cache = 256;
			break;
		case 0x83:
			CPU_Info.L2Cache = 512;
			break;
		case 0x84:
			CPU_Info.L2Cache = 1024;
			break;
		case 0x85:
			CPU_Info.L2Cache = 2048;
			break;
		case 0x86:
			CPU_Info.L2Cache = 512;
			break;
		case 0x87:
			CPU_Info.L2Cache = 1024;
			break;
		// L3 Cache
		case 0x22:
			CPU_Info.L3Cache = 512;
			break;
		case 0x23:
			CPU_Info.L3Cache = 1024;
			break;
		case 0x25:
			CPU_Info.L3Cache = 2048;
			break;
		case 0x29:
			CPU_Info.L3Cache = 4096;
			break;
		case 0x88:
			CPU_Info.L3Cache = 2048;
			break;
		case 0x89:
			CPU_Info.L3Cache = 4096;
			break;
		case 0x8A:
			CPU_Info.L3Cache = 8192;
			break;
		case 0x8D:
			CPU_Info.L3Cache = 3096;
			break;
	}
}

/******************************************************************************

 Copyright (c) 1999 Advanced Micro Devices, Inc.

 LIMITATION OF LIABILITY:  THE MATERIALS ARE PROVIDED *AS IS* WITHOUT ANY
 EXPRESS OR IMPLIED WARRANTY OF ANY KIND INCLUDING WARRANTIES OF MERCHANTABILITY,
 NONINFRINGEMENT OF THIRD-PARTY INTELLECTUAL PROPERTY, OR FITNESS FOR ANY
 PARTICULAR PURPOSE.  IN NO EVENT SHALL AMD OR ITS SUPPLIERS BE LIABLE FOR ANY
 DAMAGES WHATSOEVER (INCLUDING, WITHOUT LIMITATION, DAMAGES FOR LOSS OF PROFITS,
 BUSINESS INTERRUPTION, LOSS OF INFORMATION) ARISING OUT OF THE USE OF OR
 INABILITY TO USE THE MATERIALS, EVEN IF AMD HAS BEEN ADVISED OF THE POSSIBILITY
 OF SUCH DAMAGES.  BECAUSE SOME JURISDICTIONS PROHIBIT THE EXCLUSION OR LIMITATION
 OF LIABILITY FOR CONSEQUENTIAL OR INCIDENTAL DAMAGES, THE ABOVE LIMITATION MAY
 NOT APPLY TO YOU.

 AMD does not assume any responsibility for any errors which may appear in the
 Materials nor any responsibility to support or update the Materials.  AMD retains
 the right to make changes to its test specifications at any time, without notice.

 NO SUPPORT OBLIGATION: AMD is not obligated to furnish, support, or make any
 further information, software, technical information, know-how, or show-how
 available to you.

 So that all may benefit from your experience, please report  any  problems
 or  suggestions about this software to 3dsdk.support@amd.com

 AMD Developer Technologies, M/S 585
 Advanced Micro Devices, Inc.
 5900 E. Ben White Blvd.
 Austin, TX 78741
 3dsdk.support@amd.com
******************************************************************************/
//
/********************************************************
 Function: cpuTimeStamp (assume pentium)
 RDTSC returns 64 bits of data in EAX,EDX
 We only use the lower 32 bits so, an overflow could occur every 8 seconds.
********************************************************/
unsigned int CProcessorInfo::TimeStamp()
{

	unsigned int TickVal;

	__asm
	{	 
		_emit 0Fh
		_emit 31h
		mov	  TickVal, eax
	}

	return TickVal;	
}	

/********************************************************
PROCESSOR SPEED TEST 
********************************************************/
void CProcessorInfo::SetSpeed(void)
{
	SetPriorityClass(GetCurrentProcess(), HIGH_PRIORITY_CLASS);
	//
	int				timeStart	= 0;
	int				timeStop	= 0;
	unsigned long   StartTicks	= 0;
	unsigned long   EndTicks	= 0;
	unsigned long   TotalTicks	= 0;

	timeStart = timeGetTime();				//GET TICK EDGE
	for(;;)
	{
		timeStop = timeGetTime();		
		if ( (timeStop-timeStart) > 1 )		// ROLLOVER PAST 1
		{
			__asm
			{
				 xor    eax, eax
				 xor    ebx, ebx
				 xor    ecx, ecx
				 xor    edx, edx
				 _emit  0x0f				// CPUID 
				 _emit  0xa2
				 _emit  0x0f				// RDTSC
				 _emit  0x31
				 mov    [StartTicks], eax	// TICK COUNTER STARTS HERE
			}
			break;	
		}	
	}
	timeStart = timeStop;				

	for(;;)
	{
		timeStop = timeGetTime();			
		if ( (timeStop-timeStart) > 1000 )	//ONE SECOND
		{
			__asm
			{
				 xor    eax, eax
				 xor    ebx, ebx
				 xor    ecx, ecx
				 xor    edx, edx
				 _emit  0x0f				// CPUID 
				 _emit  0xa2
				 _emit  0x0f				// RDTSC
				 _emit  0x31
				 mov    [EndTicks], eax		// TICK COUNTER STARTS HERE
			}				
			break;	
		}	
	}

	TotalTicks = EndTicks-StartTicks;		// TOTAL 

	CPU_Info.Speed = unsigned int(TotalTicks / 1000000);	// SPEED
	//
	SetPriorityClass(GetCurrentProcess(), NORMAL_PRIORITY_CLASS);
}

/******************************************************************************
 DATA CACHE
 GET LEVEL 1 AND TEST TO SEE IF LEVEL 2 CACHE INFORMATION IS AVAILABLE
 IF SO, GET LEVEL 2 CACHE SIZE
******************************************************************************/
/*
void CProcessorInfo::SetCacheAMD()
{
    unsigned __int32	L1CacheCode = 0;
	unsigned __int32	L1CacheData = 0;
	unsigned __int32	L2Cache = 0;
	//
	__asm
	{
		mov		eax, 0x80000005				// L1 CACHE
		mov		ebx, 0
		mov		ecx, 0
		mov		edx, 0

		_emit	0x0f
		_emit	0xa2

		shr		ecx, 24
		mov		L1CacheData, ecx
		shr		edx, 24
		mov		L1CacheCode, edx
	}
	CPU_Info.L1CacheData = L1CacheData;
	CPU_Info.L1CacheCode = L1CacheCode;

//	if (largest >= 0x80000006)				// SEE IF WE CAN ASK FOR THIS ONE, largest extended function 8000000X
//	{
		__asm
		{
			mov		eax, 0x80000006			// L2 CACHE
			mov		ebx, 0
			mov		ecx, 0
			mov		edx, 0

			_emit	0x0f
			_emit	0xa2

			shr		ecx, 16
			mov		L2Cache, ecx
		}
//	}
	CPU_Info.L2Cache = L2Cache;
}
*/