//
#include "stdafx.h"
#include <math.h>
#include "emule.h"
#include "ClientCredits.h"
#include "OtherFunctions.h"
#include "Preferences.h"
#include "SafeFile.h"
#include "Opcodes.h"
#include "Sockets.h"
#include <crypto51/base64.h>
#include <crypto51/osrng.h>
#include <crypto51/files.h>
#include <crypto51/sha.h>
#include "emuledlg.h"
#include "PartFile.h"
#include "Log.h"
#include "0RatioFile/RT_Opcodes.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define CLIENTS_MET_FILENAME		_T("clients.met")
#define RT_CLIENTS_MET_FILENAME		_T("RT_Clients.dat")

CClientCredits::CClientCredits(CreditStruct* in_credits)
{
	m_pCredits = in_credits;
	InitalizeIdent();
}

CClientCredits::CClientCredits(const uchar* key)
{
	m_pCredits = new CreditStruct;
	memset( m_pCredits, 0, sizeof(CreditStruct) );
	md4cpy(m_pCredits->abyKey, key);
	InitalizeIdent();
}

CClientCredits::~CClientCredits()
{
	if (m_abyPublicKey != m_pCredits->abySecureIdent)   delete [] m_abyPublicKey;
	delete m_pCredits;
}

void CClientCredits::InitalizeIdent()
{
	if (m_pCredits->nKeySize == 0 )
	{
		m_abyPublicKey = new uchar[MAXPUBKEYSIZE];
		memset(m_abyPublicKey, 0, MAXPUBKEYSIZE); // for debugging
		m_nPublicKeyLen = 0;
		IdentState = IS_NOTAVAILABLE;
	}
	else
	{
		m_nPublicKeyLen = m_pCredits->nKeySize;
		m_abyPublicKey = (uchar*)m_pCredits->abySecureIdent;
		IdentState = IS_IDNEEDED;
	}
	//
	m_dwCryptRndChallengeFor = 0;
	m_dwCryptRndChallengeFrom = 0;
	m_dwIdentIP = 0;
}

void CClientCredits::Verified(uint32 dwForIP)
{
	m_dwIdentIP = dwForIP;
	// client was verified, copy the keyto store him if not done already
	if (m_pCredits->nKeySize == 0)
	{
		m_pCredits->nKeySize = m_nPublicKeyLen; 
		memcpy(m_pCredits->abySecureIdent, m_abyPublicKey, m_nPublicKeyLen);
		// for security reason, we have to delete all prior credits here
		if ( (GetDownloadedTotal() > 0) || (GetUploadedTotal() > 0) )
		{
			m_pCredits->DownloadedTotal = 0;
			m_pCredits->UploadedTotal = 0;
			if (thePrefs.GetVerbose() == true)
				DEBUG_ONLY(AddDebugLogLine(false, _T("Credits deleted due to new SecureIdent")));
		}
		// Save RAM
		if ( (m_abyPublicKey != NULL) && (m_abyPublicKey != m_pCredits->abySecureIdent) )   delete [] m_abyPublicKey;
		m_abyPublicKey = m_pCredits->abySecureIdent;
	}
	IdentState = IS_IDENTIFIED;
}

bool CClientCredits::SetSecureIdent(uchar* pachIdent, uint8 nIdentLen)
{
	// verified Public key cannot change, use only if there is not public key yet
	if ( (MAXPUBKEYSIZE < nIdentLen) || (m_pCredits->nKeySize != 0) )   return false;
	memcpy(m_abyPublicKey, pachIdent, nIdentLen);
	m_nPublicKeyLen = nIdentLen;
	IdentState = IS_IDNEEDED;
	return true;
}

EIdentState	CClientCredits::GetCurrentIdentState(uint32 dwForIP) const
{
	if (IdentState != IS_IDENTIFIED)
		return IdentState;
	else
	{
		if (dwForIP == m_dwIdentIP)
			return IS_IDENTIFIED;
		else
			return IS_IDBADGUY; 
			// mod note: clients which just reconnected after an IP change and have to ident yet will also have this state for 1-2 seconds
			//		 so don't try to spam such clients with "bad guy" messages (besides: spam messages are always bad)
	}
}

// Use Uint64
void CClientCredits::AddDownloaded(uint32 bytes, uint32 dwForIP)
{
	// check the client ident status
	if (theApp.clientcredits->CryptoAvailable() == true)
	{
		switch ( GetCurrentIdentState(dwForIP) )
		{
			case IS_IDFAILED:
			case IS_IDNEEDED:
			case IS_IDBADGUY:
				return;
		}
	}
	m_pCredits->DownloadedTotal += bytes;
}

// Use Uint64
void CClientCredits::AddUploaded(uint32 bytes, uint32 dwForIP)
{
	// check the client ident status
	if (theApp.clientcredits->CryptoAvailable() == true)
	{
		switch ( GetCurrentIdentState(dwForIP) )
		{
			case IS_IDFAILED:
			case IS_IDNEEDED:
			case IS_IDBADGUY:
				return;
		}
	}
	m_pCredits->UploadedTotal += bytes;
}

// Use Uint64
uint64	CClientCredits::GetUploadedTotal() const
{
	return m_pCredits->UploadedTotal;
}

// Use Uint64
uint64	CClientCredits::GetDownloadedTotal() const
{
	return m_pCredits->DownloadedTotal;
}

// New Credit System
float CClientCredits::GetScoreRatio(uint32 dwForIP) const
{
	// check the client ident status
	if (theApp.clientcredits->CryptoAvailable() == true)
	{
		switch ( GetCurrentIdentState(dwForIP) )
		{
			case IS_IDFAILED:
			case IS_IDNEEDED:
			case IS_IDBADGUY:
				return 0;
		}
	}
	//
	float UploadedTotalMB = float(GetUploadedTotal()) / 1048576;
	float DownloadedTotalMB = float(GetDownloadedTotal()) / 1048576;
	float Result;
	//
	if (DownloadedTotalMB <= 1)
	{
		if (UploadedTotalMB <= 1)
			Result = 1;
		else
		{
			if (UploadedTotalMB <= 18)
				Result = 1 - (UploadedTotalMB / 20);
			else
				Result = 0.1 - ((UploadedTotalMB - 18) / 300);
			if (Result < RT_CREDIT_RATIO_LOW_LIMIT)   Result = RT_CREDIT_RATIO_LOW_LIMIT;
		}
	}
	else if (UploadedTotalMB > 1)
	{
		float DownloadedTotalSQRT = sqrt(DownloadedTotalMB);
		Result = 2 + (DownloadedTotalSQRT / 10);
		if (DownloadedTotalMB > UploadedTotalMB)
		{
			Result += sqrt(DownloadedTotalMB - UploadedTotalMB);
		}
		else
		{
			float Difference = UploadedTotalMB - DownloadedTotalMB;
			if (Difference > 1)
			{
				if (Difference <= 19)
					Result -= Difference / 10;
				else
					Result = 0.1 - ((Difference - 19) / 300);
				if ( (DownloadedTotalMB * 4) > UploadedTotalMB )
				{
					float Lowest = 1 + (DownloadedTotalSQRT / 10);
					if (Lowest > 2)   Lowest = 2;
					if (Result < Lowest)   Result = Lowest;
				}
				else
				{
					if (Result < 1)
					{
						if ( (DownloadedTotalMB * DownloadedTotalSQRT) > UploadedTotalMB )
							Result = 1;
						else if (Result <= RT_CREDIT_RATIO_LOW_LIMIT)
							Result = RT_CREDIT_RATIO_LOW_LIMIT;
					}
					else
						Result = 1;
				}
				// Credit1 Slot
				if ( (Result > 1) && (thePrefs.GetMaxCredit1Slot() > 0) && 
					(UploadedTotalMB - DownloadedTotalMB) > max((DownloadedTotalSQRT * 2), 10) )
				{
					Result = 1;
				}
			}
		}
	}
	else
	{
		Result = DownloadedTotalMB + 1;
	}
	return Result;
}

// Don't Download
bool CClientCredits::IsDontDownload() const
{
	return (m_pCredits->Feature & RT_CREDIT_DONT_DOWNLOAD);
}

// Don't Download
void CClientCredits::SetDontDownload(bool NewState)
{
	if (NewState == true)
		m_pCredits->Feature |= RT_CREDIT_DONT_DOWNLOAD;
	else if (m_pCredits->Feature & RT_CREDIT_DONT_DOWNLOAD)
		m_pCredits->Feature ^= RT_CREDIT_DONT_DOWNLOAD;
}

// Leecher Count
uint8 CClientCredits::GetLeecherCount() const
{
	return (m_pCredits->Feature & RT_CREDIT_LEECHER_COUNT);
}

// Leecher Count
void CClientCredits::AddLeecherCount()
{
	if (GetLeecherCount() < RT_LEECHER_COUNT_MAX)   m_pCredits->Feature++;
}

// Leecher Count
void CClientCredits::SetLeecherCount(uint8 NewValue)
{
	uint32 Mask = 0xFFFFFFFF ^ RT_CREDIT_LEECHER_COUNT;
	m_pCredits->Feature = (m_pCredits->Feature & Mask) + NewValue;
}

/***************
* Credits List *
****************/
//
CClientCreditsList::CClientCreditsList()
{
	m_nLastSaved = ::GetTickCount();
	LoadList();
	InitalizeCrypting();
}

CClientCreditsList::~CClientCreditsList()
{
	SaveList();
	SaveList2ClientsMet();
	CClientCredits* ClientCredit;
	CCKey tmpkey(0);
	POSITION pos = m_mapClients.GetStartPosition();
	while (pos)
	{
		m_mapClients.GetNextAssoc(pos, tmpkey, ClientCredit);
		delete ClientCredit;
	}
	delete m_pSignkey;
}

void CClientCreditsList::LoadList()
{
	CString Filename = thePrefs.GetConfigDir() + CString(RT_CLIENTS_MET_FILENAME);
	const int iOpenFlags = CFile::modeRead|CFile::osSequentialScan|CFile::typeBinary|CFile::shareDenyWrite;
	CSafeBufferedFile CreditsFile;
	CFileException fexp;
	// Open File
	if (CreditsFile.Open(Filename, iOpenFlags, &fexp) == FALSE)
	{
		if (fexp.m_cause != CFileException::fileNotFound)
		{
			CString strError(GetResString(IDS_ERR_LOADCREDITFILE));
			TCHAR szError[MAX_CFEXP_ERRORMSG];
			if (fexp.GetErrorMessage(szError, ARRSIZE(szError)) == TRUE)
			{
				strError += _T(" - ");
				strError += szError;
			}
			AddLogLine(true, _T("%s"), strError);
		}
		return;
	}
	setvbuf(CreditsFile.m_pStream, NULL, _IOFBF, 16384);
	//
	try
	{
		// Check Version
		uint8 CreditsVersion = CreditsFile.ReadUInt8();
		if (CreditsVersion != RT_CREDIT_FILE_VERSION)
		{
			AddLogLine( false, GetResString(IDS_ERR_CREDITFILEOLD) );
			CreditsFile.Close();
			return;
		}
		 // TODO: should be prime number... and 20% larger
		uint32 CreditsCount = CreditsFile.ReadUInt32();
		uint32 TableSize = CreditsCount * 1.2f;
		if (thePrefs.GetQueueSize() > CreditsCount)   TableSize = thePrefs.GetQueueSize() * 1.2f;
		m_mapClients.InitHashTable( TableSize );
		const uint32 dwExpired = time(NULL) - DAY2S(90);
		const uint32 WaitedTimeExpired = time(NULL) - DAY2S(4);
		// Load Data
		bool IsResetDontDownload = thePrefs.IsResetDontDownload();
		uint32 DeletedCount = 0;
		for (uint32 i = 0; i < CreditsCount; i++)
		{
			CreditStruct* RT_Credit = new CreditStruct;
			memset( RT_Credit, 0, sizeof(CreditStruct) );
			CreditsFile.Read( RT_Credit, sizeof(CreditStruct) );
			if (RT_Credit->nLastSeen < dwExpired)
			{
				DeletedCount++;
				delete RT_Credit;
				continue;
			}
			// Upload Queue Waited Time
			if (RT_Credit->UploadQueueWaitedTime > 0)
			{
				if ( (RT_Credit->nLastSeen < WaitedTimeExpired) || (thePrefs.IsRecordUploadQueueWaitedTime() == false) )
				{
					if ( (RT_Credit->DownloadedTotal < 1048576) && (RT_Credit->UploadedTotal < 1048576) )
					{
						DeletedCount++;
						delete RT_Credit;
						continue;
					}
					RT_Credit->UploadQueueWaitedTime = 0;
				}
			}
			//
			CClientCredits* ClientCredit = new CClientCredits(RT_Credit);
			m_mapClients.SetAt( CCKey(ClientCredit->GetKey()), ClientCredit );
			// Reset DontDownload
			if (IsResetDontDownload == true)   ClientCredit->SetDontDownload(false);
		}
		CreditsFile.Close();
		//
		if (DeletedCount > 0)
			AddLogLine(false, GetResString(IDS_CREDITFILELOADED) + GetResString(IDS_CREDITSEXPIRED), (CreditsCount - DeletedCount), DeletedCount);
		else
			AddLogLine(false, GetResString(IDS_CREDITFILELOADED), CreditsCount);

	}
	catch (CFileException* error)
	{
		if (error->m_cause == CFileException::endOfFile)
			AddLogLine( true, GetResString(IDS_CREDITFILECORRUPT) );
		else
		{
			TCHAR Buffer[MAX_CFEXP_ERRORMSG];
			error->GetErrorMessage( Buffer, ARRSIZE(Buffer) );
			AddLogLine(true, GetResString(IDS_ERR_CREDITFILEREAD), Buffer);
		}
		error->Delete();
		return;
	}
}

void CClientCreditsList::SaveList()
{
	// Test
//	if (thePrefs.IsLogRatioVerbose() == true)
//		AddDebugLogLine( false, _T(">> [RT Debug] The number of ClientCredits is %u"), m_mapClients.GetCount() );
	//
	if (thePrefs.GetLogFileSaving() == true)
		AddDebugLogLine(false, _T("Saving clients credit list file \"%s\""), RT_CLIENTS_MET_FILENAME);
	m_nLastSaved = ::GetTickCount();

	CString Filename = thePrefs.GetConfigDir() + CString(RT_CLIENTS_MET_FILENAME);
	CFile CreditsFile;// no buffering needed here since we swap out the entire array
	CFileException fexp;
	// Backup
	CString BackupFile = Filename + CString( _T(".bak") );
	if (PathFileExists(Filename) == TRUE)
	{
		if (PathFileExists(BackupFile) == TRUE)   _tremove(BackupFile);
		_trename(Filename, BackupFile);
	}
	// Open
	if (CreditsFile.Open(Filename, CFile::modeWrite|CFile::modeCreate|CFile::typeBinary|CFile::shareDenyWrite, &fexp) == FALSE)
	{
		CString strError( GetResString(IDS_ERR_FAILED_CREDITSAVE) );
		TCHAR szError[MAX_CFEXP_ERRORMSG];
		if (fexp.GetErrorMessage(szError, ARRSIZE(szError)) == TRUE)
		{
			strError += _T(" - ");
			strError += szError;
		}
		AddLogLine(true, _T("%s"), strError);
		return;
	}
	// Flag of Normal Exit
	thePrefs.SetNormalExit( _T("RT_Clients"), false );
	//
	uint32 CreditsCount = m_mapClients.GetCount();
	BYTE* pBuffer = new BYTE[ (CreditsCount * sizeof(CreditStruct)) ];
	const uint32 WaitedTimeExpired = time(NULL) - DAY2S(4);
	//
	CreditsCount = 0;
	uint32 BufferPos = 0;
	CCKey TempKey(0);
	CClientCredits* ClientCredit;
	POSITION MapPos = m_mapClients.GetStartPosition();
	while (MapPos != NULL)
	{
		m_mapClients.GetNextAssoc(MapPos, TempKey, ClientCredit);
		// Upload Queue Waited Time
		if (ClientCredit->GetUploadQueueWaitedTime() > 0)
		{
			if ( (thePrefs.IsRecordUploadQueueWaitedTime() == false) ||
				(ClientCredit->m_pCredits->nLastSeen < WaitedTimeExpired) ||
				((ClientCredit->GetLeecherCount() * EMBLOCKSIZE) > ClientCredit->GetUploadedTotal()) )
			{
				ClientCredit->SetUploadQueueWaitedTime(0);
			}
		}
		// Write to Buffer
		if ( (ClientCredit->GetUploadedTotal() > 1048576) || (ClientCredit->GetDownloadedTotal() > 1048576) ||
			(ClientCredit->m_pCredits->Feature > 0) || (ClientCredit->GetUploadQueueWaitedTime() > 0) )
		{
			memcpy( (pBuffer + BufferPos), ClientCredit->GetDataStruct(), sizeof(CreditStruct) );
			CreditsCount++;
			BufferPos += sizeof(CreditStruct);
		}
	}
	//
	try
	{
		// Save
		uint8 CreditsVersion = RT_CREDIT_FILE_VERSION;
		CreditsFile.Write(&CreditsVersion, 1);
		CreditsFile.Write(&CreditsCount, 4);
		CreditsFile.Write( pBuffer, (CreditsCount * sizeof(CreditStruct)) );
		if ( (thePrefs.GetCommitFiles() >= 2) || ((thePrefs.GetCommitFiles() >= 1) && (theApp.emuledlg->IsRunning() == false)) )
			CreditsFile.Flush();
		CreditsFile.Close();
	}
	catch (CFileException* error)
	{
		CString strError(GetResString(IDS_ERR_FAILED_CREDITSAVE));
		TCHAR szError[MAX_CFEXP_ERRORMSG];
		if (error->GetErrorMessage(szError, ARRSIZE(szError)) == TRUE)
		{
			strError += _T(" - ");
			strError += szError;
		}
		AddLogLine(true, _T("%s"), strError);
		error->Delete();
		// Delete File
		if (PathFileExists(Filename) == TRUE)   _tremove(Filename);
		// Restore File
		if (PathFileExists(BackupFile) == TRUE)   _trename(BackupFile, Filename);
	}
	//
	delete[] pBuffer;
	// Flag of Normal Exit
	thePrefs.SetNormalExit( _T("RT_Clients"), true );
}

void CClientCreditsList::SaveList2ClientsMet()
{
	//
	CString name = thePrefs.GetConfigDir() + CString(CLIENTS_MET_FILENAME);
	CFile file;// no buffering needed here since we swap out the entire array
	CFileException fexp;
	if (file.Open(name, CFile::modeWrite|CFile::modeCreate|CFile::typeBinary|CFile::shareDenyWrite, &fexp) == FALSE)
	{
		CString strError(GetResString(IDS_ERR_FAILED_CREDITSAVE));
		TCHAR szError[MAX_CFEXP_ERRORMSG];
		if (fexp.GetErrorMessage(szError, ARRSIZE(szError)) == TRUE)
		{
			strError += _T(" - ");
			strError += szError;
		}
		AddLogLine(true, _T("%s"), strError);
		return;
	}
	//
	CClientCredits* ClientCredit;
	CreditStruct_Official* OfficialCredit = new CreditStruct_Official;
	CCKey tempkey(0);
	uint32 count = m_mapClients.GetCount();
	POSITION pos = m_mapClients.GetStartPosition();
	BYTE* pBuffer = new BYTE[count*sizeof(CreditStruct_Official)];
	count = 0;
	uint32 BufferPos = 0;
	// Write to Buffer
	while (pos != NULL)
	{
		m_mapClients.GetNextAssoc(pos, tempkey, ClientCredit);
		if ( (ClientCredit->GetUploadedTotal() > EMBLOCKSIZE) || (ClientCredit->GetDownloadedTotal() > EMBLOCKSIZE) ||
			(ClientCredit->m_pCredits->Feature > 0) )
		{
			OfficialCredit->nUploadedLo = uint32(ClientCredit->m_pCredits->UploadedTotal);
			OfficialCredit->nUploadedHi = uint32(ClientCredit->m_pCredits->UploadedTotal >> 32);
			OfficialCredit->nDownloadedLo = uint32(ClientCredit->m_pCredits->DownloadedTotal);
			OfficialCredit->nDownloadedHi = uint32(ClientCredit->m_pCredits->DownloadedTotal >> 32);
			OfficialCredit->nReserved3 = 0;
			OfficialCredit->nLastSeen = ClientCredit->m_pCredits->nLastSeen;
			OfficialCredit->nKeySize = ClientCredit->m_pCredits->nKeySize;
			memcpy( OfficialCredit->abyKey, ClientCredit->m_pCredits->abyKey, sizeof(OfficialCredit->abyKey) );
			memcpy( OfficialCredit->abySecureIdent, ClientCredit->m_pCredits->abySecureIdent, sizeof(OfficialCredit->abySecureIdent) );
			//
			memcpy( (pBuffer + BufferPos), OfficialCredit, sizeof(CreditStruct_Official) );
			count++;
			BufferPos += sizeof(CreditStruct_Official);
		}
	}
	delete [] OfficialCredit;
	// Save
	try
	{
		uint8 version = CREDITFILE_VERSION;
		file.Write(&version, 1);
		file.Write(&count, 4);
		file.Write(pBuffer, count*sizeof(CreditStruct_Official));
		file.Flush();
		file.Close();
	}
	catch (CFileException* error)
	{
		CString strError(GetResString(IDS_ERR_FAILED_CREDITSAVE));
		TCHAR szError[MAX_CFEXP_ERRORMSG];
		if (error->GetErrorMessage(szError, ARRSIZE(szError)) == TRUE)
		{
			strError += _T(" - ");
			strError += szError;
		}
		AddLogLine(true, _T("%s"), strError);
		error->Delete();
	}
	delete[] pBuffer;
}

// RT, New Code at Bottom
/* Original
CClientCredits* CClientCreditsList::GetCredit(const uchar* key)
{
	CClientCredits* result;
	CCKey tkey(key);
	if (m_mapClients.Lookup(tkey, result) == FALSE)
	{
		result = new CClientCredits(key);
		m_mapClients.SetAt(CCKey(result->GetKey()), result);
	}
	result->SetLastSeen();
	return result;
}
*/

void CClientCreditsList::Process()
{
	if ( (::GetTickCount() - m_nLastSaved) > MIN2MS(13) )   SaveList();
}

using namespace CryptoPP;

void CClientCreditsList::InitalizeCrypting()
{
	m_nMyPublicKeyLen = 0;
	memset(m_abyMyPublicKey, 0, 80); // not really needed; better for debugging tho
	m_pSignkey = NULL;
	if (thePrefs.IsSecureIdentEnabled() == false)   return;
	// check if keyfile is there
	bool bCreateNewKey = false;
	HANDLE hKeyFile = ::CreateFile(thePrefs.GetConfigDir() + CString("cryptkey.dat"), GENERIC_READ, FILE_SHARE_READ, NULL,
										OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
	if (hKeyFile != INVALID_HANDLE_VALUE)
	{
		if (::GetFileSize(hKeyFile, NULL) == 0)   bCreateNewKey = true;
		::CloseHandle(hKeyFile);
	}
	else
		bCreateNewKey = true;
	//
	if (bCreateNewKey)   CreateKeyPair();
	// load key
	try
	{
		// load private key
		FileSource filesource(CStringA(thePrefs.GetConfigDir() + _T("cryptkey.dat")), true,new Base64Decoder);
		m_pSignkey = new RSASSA_PKCS1v15_SHA_Signer(filesource);
		// calculate and store public key
		RSASSA_PKCS1v15_SHA_Verifier pubkey(*m_pSignkey);
		ArraySink asink(m_abyMyPublicKey, 80);
		pubkey.DEREncode(asink);
		m_nMyPublicKeyLen = asink.TotalPutLength();
		asink.MessageEnd();
		// RSA-CIDE by enkeyDev
		if ( (bCreateNewKey == true) || (thePrefs.IsNewUserHash() == true) )
		{
			uchar NewUserHash[16];
			MD4HashThis(m_abyMyPublicKey, m_nMyPublicKeyLen, NewUserHash);
			NewUserHash[5] = 14;
			NewUserHash[14] = 111;
			thePrefs.SetUserHash(NewUserHash);
			if (bCreateNewKey == true)
				theApp.QueueLogLine( true, _T("UserHash Created by RSA-CIDE Method") );
			else
				theApp.QueueLogLine( true, _T("UserHash Recovered by RSA-CIDE Method") );
		}
// End
	}
	catch(...)
	{
		if (m_pSignkey)
		{
			delete m_pSignkey;
			m_pSignkey = NULL;
		}
		AddLogLine( false, GetResString(IDS_CRYPT_INITFAILED) );
		ASSERT(0);
	}
	//Debug_CheckCrypting();
}

bool CClientCreditsList::CreateKeyPair()
{
	try
	{
		AutoSeededRandomPool rng;
		InvertibleRSAFunction privkey;
		privkey.Initialize(rng,RSAKEYSIZE);

		Base64Encoder privkeysink( new FileSink(CStringA(thePrefs.GetConfigDir() + _T("cryptkey.dat"))) );
		privkey.DEREncode(privkeysink);
		privkeysink.MessageEnd();

		if (thePrefs.GetLogSecureIdent() == true)   AddDebugLogLine(false, _T("Created new RSA keypair"));
	}
	catch(...)
	{
		if (thePrefs.GetVerbose() == true)   AddDebugLogLine(false, _T("Failed to create new RSA keypair"));
		ASSERT ( false );
		return false;
	}
	return true;
}

uint8 CClientCreditsList::CreateSignature(CClientCredits* pTarget, uchar* pachOutput, uint8 nMaxSize, uint32 ChallengeIP, uint8 byChaIPKind, CryptoPP::RSASSA_PKCS1v15_SHA_Signer* sigkey)
{
	// sigkey param is used for debug only
	if (sigkey == NULL)   sigkey = m_pSignkey;

	// create a signature of the public key from pTarget
	ASSERT( pTarget );
	ASSERT( pachOutput );
	uint8 nResult;
	if (CryptoAvailable() == false)   return 0;
	try
	{		
		SecByteBlock sbbSignature( sigkey->SignatureLength() );
		AutoSeededRandomPool rng;
		byte abyBuffer[MAXPUBKEYSIZE+9];
		uint32 keylen = pTarget->GetSecIDKeyLen();
		memcpy(abyBuffer, pTarget->GetSecureIdent(), keylen);
		// 4 additional bytes random data send from this client
		uint32 challenge = pTarget->m_dwCryptRndChallengeFrom;
		ASSERT ( challenge != 0 );
		PokeUInt32(abyBuffer+keylen, challenge);
		uint16 ChIpLen = 0;
		if ( byChaIPKind != 0)
		{
			ChIpLen = 5;
			PokeUInt32(abyBuffer+keylen+4, ChallengeIP);
			PokeUInt8(abyBuffer+keylen+4+4, byChaIPKind);
		}
		sigkey->SignMessage( rng, abyBuffer, keylen+4+ChIpLen , sbbSignature.begin() );
		ArraySink asink(pachOutput, nMaxSize);
		asink.Put( sbbSignature.begin(), sbbSignature.size() );
		nResult = asink.TotalPutLength();			
	}
	catch(...)
	{
		ASSERT ( false );
		nResult = 0;
	}
	return nResult;
}

bool CClientCreditsList::VerifyIdent(CClientCredits* pTarget, uchar* pachSignature, uint8 nInputSize, uint32 dwForIP, uint8 byChaIPKind)
{
	if (pTarget == NULL)
	{
		if (thePrefs.IsLogRatioVerbose() == true)
			AddDebugLogLine( false, _T(">> [RT Debug] ClientCredits: NULL, CClientCreditsList::VerifyIdent()") );
		return false;
	}
	if (pachSignature == NULL)
	{
		if (thePrefs.IsLogRatioVerbose() == true)
			AddDebugLogLine( false, _T(">> [RT Debug] Signature: NULL, CClientCreditsList::VerifyIdent()") );
		return false;
	}
	ASSERT( pTarget );
	ASSERT( pachSignature );
	if (CryptoAvailable() == false)
	{
		pTarget->IdentState = IS_NOTAVAILABLE;
		return false;
	}
	bool bResult;
	try
	{
		StringSource ss_Pubkey( (byte*)pTarget->GetSecureIdent(), pTarget->GetSecIDKeyLen(), true, 0 );
		RSASSA_PKCS1v15_SHA_Verifier pubkey(ss_Pubkey);
		// 4 additional bytes random data send from this client +5 bytes v2
		byte abyBuffer[MAXPUBKEYSIZE+9];
		memcpy(abyBuffer, m_abyMyPublicKey, m_nMyPublicKeyLen);
		uint32 challenge = pTarget->m_dwCryptRndChallengeFor;
		ASSERT ( challenge != 0 );
		PokeUInt32( (abyBuffer + m_nMyPublicKeyLen), challenge );
		
		// v2 security improvments (not supported by 29b, not used as default by 29c)
		uint8 nChIpSize = 0;
		if (byChaIPKind != 0)
		{
			nChIpSize = 5;
			uint32 ChallengeIP = 0;
			switch (byChaIPKind)
			{
				case CRYPT_CIP_LOCALCLIENT:
					ChallengeIP = dwForIP;
					break;
				case CRYPT_CIP_REMOTECLIENT:
					if (theApp.serverconnect->GetClientID() == 0 || theApp.serverconnect->IsLowID())
					{
						if (thePrefs.GetLogSecureIdent() == true)
							AddDebugLogLine(false, _T("Warning: Maybe SecureHash Ident fails because LocalIP is unknown"));
						ChallengeIP = theApp.serverconnect->GetLocalIP();
					}
					else
						ChallengeIP = theApp.serverconnect->GetClientID();
					break;
				case CRYPT_CIP_NONECLIENT: // maybe not supported in future versions
					ChallengeIP = 0;
					break;
			}
			PokeUInt32( (abyBuffer + m_nMyPublicKeyLen + 4), ChallengeIP );
			PokeUInt8( (abyBuffer + m_nMyPublicKeyLen + 4 + 4), byChaIPKind );
		}
		//v2 end

		bResult = pubkey.VerifyMessage(abyBuffer, m_nMyPublicKeyLen+4+nChIpSize, pachSignature, nInputSize);
	}
	catch(...)
	{
		if (thePrefs.GetVerbose() == true)
			AddDebugLogLine(false, _T("Error: Unknown exception in %hs"), __FUNCTION__);
		//ASSERT(0);
		bResult = false;
	}
	if (bResult == false)
	{
		if (pTarget->IdentState == IS_IDNEEDED)   pTarget->IdentState = IS_IDFAILED;
	}
	else
	{
		pTarget->Verified(dwForIP);
	}
	return bResult;
}

bool CClientCreditsList::CryptoAvailable()
{
	return (m_nMyPublicKeyLen > 0 && m_pSignkey != 0 && thePrefs.IsSecureIdentEnabled() );
}


#ifdef _DEBUG
bool CClientCreditsList::Debug_CheckCrypting()
{
	// create random key
	AutoSeededRandomPool rng;

	RSASSA_PKCS1v15_SHA_Signer priv(rng, 384);
	RSASSA_PKCS1v15_SHA_Verifier pub(priv);

	byte abyPublicKey[80];
	ArraySink asink(abyPublicKey, 80);
	pub.DEREncode(asink);
	uint8 PublicKeyLen = asink.TotalPutLength();
	asink.MessageEnd();
	uint32 challenge = rand();
	// create fake client which pretends to be this emule
	CreditStruct* newcstruct = new CreditStruct;
	memset(newcstruct, 0, sizeof(CreditStruct));
	CClientCredits* newcredits = new CClientCredits(newcstruct);
	newcredits->SetSecureIdent(m_abyMyPublicKey, m_nMyPublicKeyLen);
	newcredits->m_dwCryptRndChallengeFrom = challenge;
	// create signature with fake priv key
	uchar pachSignature[200];
	memset(pachSignature, 200, 0);
	uint8 sigsize = CreateSignature(newcredits, pachSignature, 200, 0, false, &priv);


	// next fake client uses the random created public key
	CreditStruct* newcstruct2 = new CreditStruct;
	memset( newcstruct2, 0, sizeof(CreditStruct) );
	CClientCredits* newcredits2 = new CClientCredits(newcstruct2);
	newcredits2->m_dwCryptRndChallengeFor = challenge;

	// if you uncomment one of the following lines the check has to fail
	//abyPublicKey[5] = 34;
	//m_abyMyPublicKey[5] = 22;
	//pachSignature[5] = 232;

	newcredits2->SetSecureIdent(abyPublicKey, PublicKeyLen);

	//now verify this signature - if it's true everything is fine
	bool bResult = VerifyIdent(newcredits2, pachSignature, sigsize, 0, 0);

	delete newcredits;
	delete newcredits2;

	return bResult;
}
#endif

/***************
* Credits List *
****************/
// 
CClientCredits* CClientCreditsList::GetCredit(const uchar* key, bool IsSetLastSeen)
{
	CClientCredits* result;
	CCKey tkey(key);
	if (m_mapClients.Lookup(tkey, result) == FALSE)
	{
		result = new CClientCredits(key);
		m_mapClients.SetAt(CCKey(result->GetKey()), result);
	}
	if (IsSetLastSeen == true)   result->SetLastSeen();
	return result;
}

bool CClientCreditsList::RemoveCredit(const uchar* UserHash, CClientCredits* Credit)
{
	CClientCredits* CurrentCredit;
	CCKey TempKey(UserHash);
	if (m_mapClients.Lookup(TempKey, CurrentCredit) == TRUE)
	{
		if (Credit == CurrentCredit)
		{
			m_mapClients.RemoveKey(TempKey);
			delete Credit;
			return true;
		}
	}
	return false;
}